package gov.va.med.mhv.sm.api.clinician.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.jaxrs.ext.MessageContext;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.api.clinician.ReAssignMessageAPIService;
import gov.va.med.mhv.sm.api.common.SessionAPIService;
import gov.va.med.mhv.sm.api.enumeration.ErrorEnum;
import gov.va.med.mhv.sm.api.transfer.ClinicianSearchFormTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianUserTO;
import gov.va.med.mhv.sm.api.transfer.CommentTO;
import gov.va.med.mhv.sm.api.transfer.CommentsTO;
import gov.va.med.mhv.sm.api.transfer.FacilityTO;
import gov.va.med.mhv.sm.api.transfer.ReAssignMessageHistoryTO;
import gov.va.med.mhv.sm.api.transfer.Session;
import gov.va.med.mhv.sm.api.transfer.TriageGroupTO;
import gov.va.med.mhv.sm.api.transfer.VisnTO;
import gov.va.med.mhv.sm.api.util.SMApiUtility;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.ReAssignMessageActionEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Annotation;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientBlockedFacility;
import gov.va.med.mhv.sm.model.PatientBlockedTriageGroup;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.PatientReassignFacility;
import gov.va.med.mhv.sm.model.ReAssignMessageHistory;
import gov.va.med.mhv.sm.model.Thread;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.service.FacilityService;
import gov.va.med.mhv.sm.service.MessageService;
import gov.va.med.mhv.sm.service.PatientBlockedService;
import gov.va.med.mhv.sm.service.ReAssignMessageService;
import gov.va.med.mhv.sm.service.ThreadService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.wsclient.queriessvc.User;

public class ReAssignMessageAPIServiceImpl implements ReAssignMessageAPIService{
	private static final Log log = LogFactory.getLog(ReAssignMessageAPIServiceImpl.class);
	
	private FacilityService facilityService;
	private SessionAPIService sessionApiService;
	private ThreadService threadService;
	private MessageService messageService;
	private AuthenticationService authenticationService;
	private TriageGroupService triageGroupService;
	private ReAssignMessageService reAssignMessageService;
	private PatientBlockedService patientBlockedService;
	private UserManagementService userManagementService;
	
	
	@Resource
	MessageContext mc;
	
	@Override
	public List<VisnTO> getVisnList() {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		
		List<VisnTO> visnTOs = new ArrayList<VisnTO>();
		CollectionServiceResponse<Facility> response = facilityService.getVisnList();
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		for (Facility fac : response.getCollection()) {
			VisnTO visnTO = new VisnTO();
			visnTO.setVisnId(fac.getId());
			visnTO.setVisnName(fac.getName());
			visnTOs.add(visnTO);
		}
		
		return visnTOs;
	}

	@Override
	public List<FacilityTO> getFicilityList(Long visnId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		
		ServiceResponse<Facility> visnResponse	=	facilityService.getFacility(visnId);
		CollectionServiceResponse<Facility> response = facilityService.getFacilitiesForVisn(visnResponse.getPayload());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		List<FacilityTO> facTOs = new ArrayList<FacilityTO>();
		Collection<Facility> coll = response.getCollection();
		// Need to remove the Facility provider is associated with
		Facility f = facilityService.getFacilityByStationNumber(c.getStationNo()).getPayload();
		if (f != null){
			for(Facility facToRemove:coll){
				if(facToRemove.getId().equals(f.getId())){
					coll.remove(facToRemove);
					break;
				}
			}
		}
		
		for (Facility fac2 : coll) {
			FacilityTO facTO = new FacilityTO();
			facTO.setFacilityId(fac2.getId());
			facTO.setFacilityName(fac2.getName());
			facTOs.add(facTO);
		}
		
		
		return facTOs;
	}

	@Override
	public List<TriageGroupTO> getinFacTriageGroupList(Long messageId) {
		return getOutFacTriageGroupList(messageId, null);
	}
	
	@Override
	public List<TriageGroupTO> getOutFacTriageGroupList(Long messageId, Long facilityId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		ServiceResponse<Patient> patResponse = authenticationService.fetchPatientById(m.getSenderId());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, patResponse);
		
		checkOwnershipAndIntegrity(c, m);
		Patient patient = patResponse.getPayload();
		String  stationNumber = null;
		if(facilityId == null)
			stationNumber = c.getStationNo();
		else {
			ServiceResponse facRes = facilityService.getFacility(facilityId);
			Facility fac = (Facility)facRes.getPayload();
			if(fac != null)
				stationNumber = fac.getStationNumber();
			else
				SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatientByStation(patient, stationNumber).getCollection();

		Collection<TriageGroup> staGroup = (Collection<TriageGroup>) triageGroupService.getTriageGroupsForStation(stationNumber).getCollection();
		
		for(TriageGroup tg:staGroup){
			if(tg.getId().equals(m.getThread().getMailGroup().getId())){
				staGroup.remove(tg);
				break;
			}
		}

		List<TriageGroupTO> tgTOs = new ArrayList<TriageGroupTO>();
		for(TriageGroup tg:staGroup){
			TriageGroupTO tgTO = new TriageGroupTO();
			tgTO.setTriageGroupId(tg.getId());
			tgTO.setTriageGroupName(tg.getName());
			for(TriageGroup pg:patGroup){
				if(tg.getId().equals(pg.getId())){
					tgTO.setPatientAssociated(true);
				}
			}
			tgTOs.add(tgTO);
			
		}
		
		return tgTOs;
	}	
	
	@Override
	public Response reassignToTGInFacility(Long messageId, Long triageGroupId, CommentTO comment) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkOwnershipAndIntegrity(c, m);
		ServiceResponse<Patient> patResponse = authenticationService.fetchPatientById(m.getSenderId());
		if (patResponse.getPayload() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148);
		}
		
		if(isPatientBlockedFromFacility(patResponse.getPayload(), triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}
		
		if(isPatientBlockedFromTG(patResponse.getPayload(), triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_IS_BLOCKED_FROM_TRIAGE_GROUP_149);
		}
		
		boolean manuallyAssociate = isPatientAssociatedWithTG(patResponse.getPayload(), triageGroupId)?
											false:true;
		Message updatedMessage = getMessageService().reAssignMessage(messageId,
				m.getRecipientId(),
				triageGroupId,
				null,
				manuallyAssociate,
				c,
				ReAssignMessageActionEnum.REASSIGN_ANOTHER_TG_WITH_IN_FACILITY).getPayload();

		if(updatedMessage==null){
			SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		
		if(comment.getText() != null) {
			addComment(m, comment, c);
		}	
		
		return Response.ok("Success").build();
	}	
	
	@Override
	public Response reassignToTGOutFacility(Long messageId, Long triageGroupId,	CommentTO comment) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkOwnershipAndIntegrity(c, m);
		Patient patient = userManagementService.findApiPatientById(m.getSenderId());
			
		if (patient == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148);
		}
		
		if(isPatientBlockedFromFacility(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}
		
		if(isPatientBlockedFromTG(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_IS_BLOCKED_FROM_TRIAGE_GROUP_149);
		}
		
		ServiceResponse<TriageGroup> tgPicked = triageGroupService.findTriageGroupById(triageGroupId);
		boolean isPatientAssociatedSelectedFacility = false;
		for(PatientFacility patFacList : patient.getFacilities() ) {
			if (patFacList.getStationNo().equals(tgPicked.getPayload().getVistaDiv())) {
				isPatientAssociatedSelectedFacility = true;
				break;
			}
		}
		
		if(isPatientBlockedFromFacility(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}
		
		if(!isPatientAssociatedSelectedFacility) {
			ServiceResponse<PatientReassignFacility> patientReassignFacility = reAssignMessageService.
					getPatientReassignFacilityByStationAndPatient(new Long(tgPicked.getPayload().getVistaDiv()),m.getSenderId());
			if(patientReassignFacility.getPayload() != null){
				return Response.ok("The patient is not associated with the selected facility."
						+ "SM Administrator has been notified to associate the patient. It's in the pending state").build();
			}
			
			Boolean notificationSent = getMessageService().nofityFacilityAdminsAddPatientToFacility(messageId,
						new Long(tgPicked.getPayload().getVistaDiv()), c).getPayload();
			if(notificationSent==Boolean.FALSE){
				SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
			}
			else {
				return Response.ok("The patient is not associated with the selected facility. "
						+ "A notification is sent to the facility's SM Administrator(s) to associate the patient. ").build();
			}
		}
		
		boolean manuallyAssociate = isPatientAssociatedWithTG(patient, triageGroupId)?
															false:true;
		Message updatedMessage = getMessageService().reAssignMessage(messageId,
				m.getRecipientId(),
				triageGroupId,
				null,
				manuallyAssociate,
				c,
				ReAssignMessageActionEnum.REASSIGN_ANOTHER_TG_WITH_IN_FACILITY).getPayload();

		if(updatedMessage==null){
			SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		
		if(comment.getText() != null )
			addComment(m, comment, c);
		
		return Response.ok("Success").build();
	}	

	@Override
	public List<ClinicianUserTO> searchClinicianInFacility(ClinicianSearchFormTO form) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		return searchClinicianInOutFacility(c, null, form);
	}

	@Override
	public List<ClinicianUserTO> searchClinicianOutFacility(Long facilityId, ClinicianSearchFormTO form) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		return searchClinicianInOutFacility(c, facilityId, form);
	}
	
	/**
	 * If we are unable to locate the user in active directory we will not include them in the results.
	 * 
	 * @param c
	 * @param facilityId
	 * @param form
	 * @return
	 */
	public List<ClinicianUserTO> searchClinicianInOutFacility(Clinician c, Long facilityId, ClinicianSearchFormTO form) {
		CollectionServiceResponse<Clinician> response = null;
		if(facilityId == null) {
			response = userManagementService.searchForClinicians(form.getFirstName(), form.getLastName(), c.getStationNo());
		}
		else {
			Facility facility = facilityService.getFacility(facilityId).getPayload();
			if(facility != null) {
				response = userManagementService.searchForClinicians(form.getFirstName(), form.getLastName(),
									(facility.getStationNumber()), true);
			}
			else {
				SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
			}
		}
		
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_MANAGEMENT_SERVICE_ERROR_906, response);
		List<ClinicianUserTO> clinicUserList = new ArrayList<ClinicianUserTO>();
		Collection<Clinician> list = response.getCollection();
		for (Clinician c1 : list) {
			ServiceResponse<User> response2 = userManagementService.getClinicianDemographics(c1);
			if( response2.getMessages().getErrorMessageCount() > 0 ) {
				log.warn("Unable to access " + c1.getUsername() + " account details from active directory");
				continue;
			}
			//SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_MANAGEMENT_SERVICE_ERROR_906, response2);
			User cd = response2.getPayload();
			ClinicianUserTO clinicTO =  
			new ClinicianUserTO(c1.getId(), c1.getUsername(), cd.getFirstName(),
					cd.getLastName(), c1.getStationNo(),
					cd.getDepartment(),cd.getPhone());
			clinicUserList.add(clinicTO);
			
		}
		return clinicUserList;
	}	
	
	@Override
	public List<TriageGroupTO> getTriageGroupsByClinicianInFac(Long messageId, Long clinicianId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkClincianExists(clinicianId);
		if(c.getId().longValue() == clinicianId) {
			// Can't Reassign to yourself..
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}	
		Patient patient = (Patient)userManagementService.findApiPatientById(m.getSenderId());
		
//		ServiceResponse<Clinician> clinicianResponse = authenticationService.authenticateClinicianById(clinicianId);
//		SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_NOT_FOUND_105, clinicianResponse);
//		Clinician clinicanToAssign = clinicianResponse.getPayload();
		Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatientByStation(patient, c.getStationNo()).getCollection();
//		Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatientByStation(patient, clinicanToAssign.getStationNo()).getCollection();	
				
		Collection<TriageGroup> clinicianGroup = triageGroupService.
				getTriageGroupsForClinician(userManagementService.fetchClinician(clinicianId).getPayload()).getCollection();
		if (clinicianGroup.size() == 0) {
			SMApiUtility.throwException(ErrorEnum.NO_TRIAGE_GROUP_FOR_CLINICIAN);
		}
		//remove the current triage group
		for(TriageGroup tg:clinicianGroup){
			if(tg.getId().equals(m.getThread().getMailGroup().getId())){
				clinicianGroup.remove(tg);
				break;
			}
		}

		List<TriageGroupTO> tgTOs = new ArrayList<TriageGroupTO>();
		for(TriageGroup tg:clinicianGroup){
			TriageGroupTO tgTO = new TriageGroupTO();
			tgTO.setTriageGroupId(tg.getId());
			tgTO.setTriageGroupName(tg.getName());
			for(TriageGroup pg:patGroup){
				if(tg.getId().equals(pg.getId())){
					tgTO.setPatientAssociated(true);
				}
			}
			tgTOs.add(tgTO);
			
		}
		
		return tgTOs;	
	}
	
	@Override
	public List<TriageGroupTO> getTriageGroupsByClinicianOutFac(Long messageId, Long clinicianId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkClincianExists(clinicianId);
		if(c.getId().longValue() == clinicianId) {
			// Can't Reassign to yourself..
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}	
		Patient patient = (Patient)userManagementService.findApiPatientById(m.getSenderId());
		
		ServiceResponse<Clinician> clinicianResponse = authenticationService.authenticateClinicianById(clinicianId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_NOT_FOUND_105, clinicianResponse);
		Clinician clinicanToAssign = clinicianResponse.getPayload();
	//	Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatientByStation(patient, c.getStationNo()).getCollection();
		Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatientByStation(patient, clinicanToAssign.getStationNo()).getCollection();	
				
		Collection<TriageGroup> clinicianGroup = triageGroupService.
				getTriageGroupsForClinician(userManagementService.fetchClinician(clinicianId).getPayload()).getCollection();
		if (clinicianGroup.size() == 0) {
			SMApiUtility.throwException(ErrorEnum.NO_TRIAGE_GROUP_FOR_CLINICIAN);
		}
		//remove the current triage group
		for(TriageGroup tg:clinicianGroup){
			if(tg.getId().equals(m.getThread().getMailGroup().getId())){
				clinicianGroup.remove(tg);
				break;
			}
		}

		List<TriageGroupTO> tgTOs = new ArrayList<TriageGroupTO>();
		for(TriageGroup tg:clinicianGroup){
			TriageGroupTO tgTO = new TriageGroupTO();
			tgTO.setTriageGroupId(tg.getId());
			tgTO.setTriageGroupName(tg.getName());
			for(TriageGroup pg:patGroup){
				if(tg.getId().equals(pg.getId())){
					tgTO.setPatientAssociated(true);
				}
			}
			tgTOs.add(tgTO);
			
		}
		
		return tgTOs;	
		
		
	}
	
	
	@Override
	public Response reassignToClinicianInFacility(Long messageId, Long triageGroupId, Long clinicianId, CommentTO comment) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		checkClincianExists(clinicianId);
		if(c.getId().longValue() == clinicianId) {
			// Can't Reassign to yourself..
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148,"Can't Reassign to yourself");
		}
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkOwnershipAndIntegrity(c, m);
		ServiceResponse<Patient> patResponse = authenticationService.fetchPatientById(m.getSenderId());
		if (patResponse.getPayload() == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148,"Message is not from a Patient" );
		}
		
		if(isPatientBlockedFromFacility(patResponse.getPayload(), triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}
		
		if(isPatientBlockedFromTG(patResponse.getPayload(), triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_IS_BLOCKED_FROM_TRIAGE_GROUP_149);
		}
		
		Collection<TriageGroup> clinicianGroup = triageGroupService.
				getTriageGroupsForClinician(userManagementService.fetchClinician(clinicianId).getPayload()).getCollection();
		if (clinicianGroup.size() == 0) {
			SMApiUtility.throwException(ErrorEnum.NO_TRIAGE_GROUP_FOR_CLINICIAN);
		}
		boolean isClinicianAssociated = false;
		for (TriageGroup tg : clinicianGroup ) {
			if (tg.getId().longValue() == triageGroupId) {
				isClinicianAssociated = true;
				break;
			}
		}
		if (!isClinicianAssociated) {
			// Clinician is not Associated
			SMApiUtility.throwException(ErrorEnum. MESSAGE_REASSIGNED_NOT_ALLOWED_148,"Clinician is not Associated");
		}
		boolean manuallyAssociate = isPatientAssociatedWithTG(patResponse.getPayload(), triageGroupId)?
															false:true;
		Message updatedMessage = getMessageService().reAssignMessage(messageId,
										m.getRecipientId(),
										triageGroupId,
										clinicianId,
										manuallyAssociate,
										c,
										ReAssignMessageActionEnum.REASSIGN_ANOTHER_STAFF_WITH_IN_FACILITY).getPayload();
		if(updatedMessage==null){
			SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		if (comment.getText() != null) {
			addComment(m, comment, c);
		}	
		
		return Response.ok("Success").build();
	}

	@Override
	public Response reassignToClinicianOutFacility(Long messageId,	Long triageGroupId, Long clinicianId, CommentTO comment) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		checkClincianExists(clinicianId);
		if(c.getId().longValue() == clinicianId) {
			// Can't Reassign to yourself..
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148,"Can't Reassign to yourself");
		}
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkOwnershipAndIntegrity(c, m);
		Patient patient = userManagementService.findApiPatientById(m.getSenderId());
		if (patient == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148);
		}
		
		if(isPatientBlockedFromFacility(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}

		if(isPatientBlockedFromTG(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_IS_BLOCKED_FROM_TRIAGE_GROUP_149);
		}
		

		ServiceResponse<TriageGroup> tgPicked = triageGroupService.findTriageGroupById(triageGroupId);
		boolean isPatientAssociatedSelectedFacility = false;
		for(PatientFacility patFacList : patient.getFacilities() ) {
			if (patFacList.getStationNo().equals(tgPicked.getPayload().getVistaDiv())) {
				isPatientAssociatedSelectedFacility = true;
			}
		}
		
		if(!isPatientAssociatedSelectedFacility) {
			ServiceResponse<PatientReassignFacility> patientReassignFacility = reAssignMessageService.
					getPatientReassignFacilityByStationAndPatient(new Long(tgPicked.getPayload().getVistaDiv()),m.getSenderId());
			if(patientReassignFacility.getPayload() != null){
				return Response.ok("The patient is not associated with the selected facility."
						+ " SM Administrator has been notfied to associate the patient. It's in the pending state").build();
			}
			
			Boolean notificationSent = getMessageService().nofityFacilityAdminsAddPatientToFacility(messageId,
						new Long(tgPicked.getPayload().getVistaDiv()), c).getPayload();
			if(notificationSent==Boolean.FALSE){
				SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
			}
			else {
				return Response.ok("The patient is not associated with the selected facility."
						+ "A notification is sent to the facility's SM Administrator(s) to associate the patient. ").build();
			}
		}
		
		Collection<TriageGroup> clinicianGroup = triageGroupService.
				getTriageGroupsForClinician(userManagementService.fetchClinician(clinicianId).getPayload()).getCollection();
		if (clinicianGroup.size() == 0) {
			SMApiUtility.throwException(ErrorEnum.NO_TRIAGE_GROUP_FOR_CLINICIAN);
		}
		boolean isClinicianAssociated = false;
		for (TriageGroup tg : clinicianGroup ) {
			if (tg.getId().longValue() == triageGroupId) {
				isClinicianAssociated = true;
				break;
			}
		}
		if (!isClinicianAssociated) {
			// Clinician is not Associated
			SMApiUtility.throwException(ErrorEnum. MESSAGE_REASSIGNED_NOT_ALLOWED_148);
		}
		boolean manuallyAssociate = isPatientAssociatedWithTG(patient, triageGroupId)?
														false:true;
		Message updatedMessage = getMessageService().reAssignMessage(messageId,
										m.getRecipientId(),
										triageGroupId,
										clinicianId,
										manuallyAssociate,
										c,
										ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_STAFF_OUTSIDE_FACILITY).getPayload();
		
		if(updatedMessage==null){
			SMApiUtility.throwException(ErrorEnum.MESSAGE_SERVICE_ERROR_904);
		}
		if (comment.getText() != null) {
			addComment(m, comment, c);
		}	
		
		return Response.ok("Success").build();		
	}

	@Override
	public List<ReAssignMessageHistoryTO> getReassignMessageHistory(Long messageId ) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		Collection<ReAssignMessageHistory> reAssignMessageList = getReAssignMessageService().getReAssignMessageHistoryByMessage(messageId);
		
		List<ReAssignMessageHistoryTO> reAssignMessageHistoryTOs = new ArrayList<ReAssignMessageHistoryTO>();
		
		for (ReAssignMessageHistory reAssignMessage : reAssignMessageList) {
			ReAssignMessageHistoryTO reAssignMessageTO = new ReAssignMessageHistoryTO(
					SMApiUtility.convertDateToRFC1123Pattern(reAssignMessage.getCreatedDate()),
					reAssignMessage.getReAssignAction().getName(),
					reAssignMessage.getTriageGroupName(),
					reAssignMessage.getReAssignedTriageGroupName(),
					reAssignMessage.getReAssignedStatus(),
					reAssignMessage.getAssignedBy(),
					reAssignMessage.getAssignedTo(),
					reAssignMessage.getCompletedBy()
					);
			
			reAssignMessageHistoryTOs.add(reAssignMessageTO);
			
		}
		
		return reAssignMessageHistoryTOs;
		
	}
	
	public Response isPatientAssociatedWithFacility(Long messageId, Long triageGroupId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		ServiceResponse<Message> response = messageService.fetchMessage(messageId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
		Message m = response.getPayload();
		checkOwnershipAndIntegrity(c, m);
		Patient patient = userManagementService.findApiPatientById(m.getSenderId());
		if (patient == null) {
			SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148);
		}
		
		if(isPatientBlockedFromFacility(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_BLOCKED_FACILITY_151);
		}

		if(isPatientBlockedFromTG(patient, triageGroupId)) {
			SMApiUtility.throwException(ErrorEnum.USER_IS_BLOCKED_FROM_TRIAGE_GROUP_149);
		}
		

		ServiceResponse<TriageGroup> tgPicked = triageGroupService.findTriageGroupById(triageGroupId);
		boolean isPatientAssociatedSelectedFacility = false;
		for(PatientFacility patFacList : patient.getFacilities() ) {
			if (patFacList.getStationNo().equals(tgPicked.getPayload().getVistaDiv())) {
				isPatientAssociatedSelectedFacility = true;
			}
		}
		return Response.ok(isPatientAssociatedSelectedFacility).build();
	}

	
	public void addComment(Message m, CommentTO comment, Clinician c) {
		
		List<CommentTO> commentList = new ArrayList<CommentTO>();
	
		if ( null != m ) {
			boolean isRecipient = false;
			for(Addressee addr:m.getAddressees()) {
				if( addr.getOwner().getId().equals(c.getId()) ) {
					isRecipient=true;
					break;
				}
			}

			if( isRecipient ) {
				Annotation na = new Annotation();
				na.setActive(true);
				na.setAnnotation(comment.getText());
				na.setCreatedDate(new Date());
				na.setAuthor(c);
				na.setThread(m.getThread());
				ServiceResponse<Boolean> response1 = messageService.annotateThread(na);
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response1);

				if( response1.getPayload().booleanValue() ) {
					//Thread mt = threadService.getCompleteThreadById(m.getThread().getId());
					Thread mt = threadService.findThreadById(m.getThread().getId());
					List<Annotation> annotationList = threadService.findAnnotationsByThreadId(mt.getId());

					for( Annotation a: annotationList) {
						commentList.add(new CommentTO(a.getId(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null),a.getAnnotation(),a.getCreatedDate()));
					}
				}
			} else {
				SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}

	}

	private boolean isPatientBlockedFromTG(Patient patient, Long triageGroupId){
		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(triageGroupId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();
		if(tg == null) 
			SMApiUtility.throwException(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902);
		
		if(patient.getStatus().equals(UserStatusEnum.BLOCKED)){
			return true;
		}
		else {
			PatientBlockedTriageGroup patBlockedTg = patientBlockedService.getPatientBlockedTriageGroupByTgIdAndPatientId(triageGroupId, patient.getId()).getPayload();
			if(patBlockedTg!=null && patBlockedTg.isBlocked()){
				return true;
			}
		}
		return false;
	}
	
	private boolean isPatientBlockedFromFacility(Patient patient, Long triageGroupId){
		
		ServiceResponse<TriageGroup> response3 = triageGroupService.findTriageGroupById(triageGroupId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902, response3);
		TriageGroup tg = response3.getPayload();
		Long vistDiv = null;
		if(tg != null)
			vistDiv = Long.valueOf(tg.getVistaDiv()).longValue();
		else 
			SMApiUtility.throwException(ErrorEnum.TRIAGEGROUP_SERVICE_ERROR_902);
		CollectionServiceResponse<PatientBlockedFacility> collectionResponse = patientBlockedService.getBlockedFacilityPatientsByPatientIdAndStation(patient.getId(), vistDiv);
		Collection<PatientBlockedFacility> patBlockedFacilities = collectionResponse.getCollection();
		if(patBlockedFacilities!=null && patBlockedFacilities.size()>0){
			return true;
		}

		return false;
	
	}
	
	private boolean isPatientAssociatedWithTG(Patient patient, Long triageGroupId) {
	
		Collection<TriageGroup> patGroup = triageGroupService.getTriageGroupsForPatient(patient).getCollection();
		
		for(TriageGroup tg : patGroup) {
			 if (tg.getId().equals(triageGroupId)) {
				 return true;
			 }
		}
		
		return false;
	}
	
	private void checkClincianExists(Long clinicianId) {
		ServiceResponse<Clinician> clinicianResponse = authenticationService.fetchClinicianById(clinicianId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, clinicianResponse);
		Clinician c1 = clinicianResponse.getPayload();
		if(null==c1) {
			SMApiUtility.throwException(ErrorEnum.USER_NOT_FOUND_105);
		}
	}
	
	
	private void checkOwnershipAndIntegrity(Clinician c, Message m) {
		boolean isRecipient = false;
		//boolean isReassigned = false;
		for(Addressee addr:m.getAddressees()) {
			if( addr.getOwner().getId().equals(c.getId()) ) {
				isRecipient=true;
				break;
			}
		}

		//CHECK IF THEY ARE THE OWNER
		if( isRecipient ) {
			Long messageFolderId = null;
			List<Addressee> list = m.getAddressees();
			for(Addressee addressee : list) {
				if( addressee.getOwner().getId().equals(c.getId()) ) {
					messageFolderId = addressee.getFolderId();
					//isReassigned  = addressee.isReAssigned(); -- REMOVED THIS CHECK
					break;
				}
			}

			if( null == messageFolderId ) {	
				SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
			}
	
			if( messageFolderId.equals(SystemFolderEnum.INBOX.getId()) || messageFolderId.equals(SystemFolderEnum.ESCALATED.getId())) {
				//if(isReassigned)
				//	SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148);
				if(!m.getSenderType().equals(ParticipantTypeEnum.PATIENT)){
					SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148, "Message doesn't originate from a Patient");
				}	
			}
			else if ( messageFolderId.equals(SystemFolderEnum.COMPLETED.getId()) || messageFolderId.equals(SystemFolderEnum.REMINDER.getId())
					|| messageFolderId.equals(SystemFolderEnum.DRAFTS.getId())	 || messageFolderId.equals(SystemFolderEnum.DELETED.getId())
					|| messageFolderId.equals(SystemFolderEnum.REASSIGN.getId()) || messageFolderId.equals(SystemFolderEnum.SENT.getId()) ) {
				
					SMApiUtility.throwException(ErrorEnum.MESSAGE_REASSIGNED_NOT_ALLOWED_148, "Message is not in an allowed folder (INBOX/ESCALATED)");
			}
			
		}
		else {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}

	}
	
	/*
	 * Getter Setter starts here
	 */
	
	public FacilityService getFacilityService() {
		return facilityService;
	}

	public void setFacilityService(FacilityService facilityService) {
		this.facilityService = facilityService;
	}

	public SessionAPIService getSessionApiService() {
		return sessionApiService;
	}

	public void setSessionApiService(SessionAPIService sessionApiService) {
		this.sessionApiService = sessionApiService;
	}

	public MessageService getMessageService() {
		return messageService;
	}

	public void setMessageService(MessageService messageService) {
		this.messageService = messageService;
	}

	public AuthenticationService getAuthenticationService() {
		return authenticationService;
	}

	public void setAuthenticationService(AuthenticationService authenticationService) {
		this.authenticationService = authenticationService;
	}

	public TriageGroupService getTriageGroupService() {
		return triageGroupService;
	}

	public void setTriageGroupService(TriageGroupService triageGroupService) {
		this.triageGroupService = triageGroupService;
	}

	public ReAssignMessageService getReAssignMessageService() {
		return reAssignMessageService;
	}

	public void setReAssignMessageService(
			ReAssignMessageService reAssignMessageService) {
		this.reAssignMessageService = reAssignMessageService;
	}

	public ThreadService getThreadService() {
		return threadService;
	}

	public void setThreadService(ThreadService threadService) {
		this.threadService = threadService;
	}

	public PatientBlockedService getPatientBlockedService() {
		return patientBlockedService;
	}

	public void setPatientBlockedService(PatientBlockedService patientBlockedService) {
		this.patientBlockedService = patientBlockedService;
	}

	public UserManagementService getUserManagementService() {
		return userManagementService;
	}

	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}


}
